/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jdt.core.tests.eval;

import junit.framework.Test;

import org.eclipse.jdt.internal.eval.GlobalVariable;
/**
 * Test the global variable evaluation.
 * This assumes that the EvaluationContext class and that the GlobalVariable class
 * are working correctly.
 */
public class VariableTest extends EvaluationTest {
/**
 * Creates a new EvaluationContextTest.
 */
public VariableTest(String name) {
	super(name);
}
public static Test suite() {
	return setupSuite(testClass());
}
/**
 * Tests the individual evaluation of variables with expected values of all kind of types.
 */
public void testAllKindOfValues() {
	try {
		// Creates the variables
		GlobalVariable var1 = context.newVariable("int".toCharArray(), "var1".toCharArray(), "1".toCharArray());
		GlobalVariable var2 = context.newVariable("boolean".toCharArray(), "var2".toCharArray(), "true".toCharArray());
		GlobalVariable var3 = context.newVariable("char".toCharArray(), "var3".toCharArray(), "'c'".toCharArray());
		GlobalVariable var4 = context.newVariable("float".toCharArray(), "var4".toCharArray(), "(float)1.0".toCharArray());
		GlobalVariable var5 = context.newVariable("double".toCharArray(), "var5".toCharArray(), "1.0".toCharArray());
		GlobalVariable var6 = context.newVariable("short".toCharArray(), "var6".toCharArray(), "(short)1".toCharArray());
		GlobalVariable var7 = context.newVariable("long".toCharArray(), "var7".toCharArray(), "(long)1".toCharArray());
		GlobalVariable var8 = context.newVariable("String".toCharArray(), "var8".toCharArray(), "\"hello\"".toCharArray());
		GlobalVariable var9 = context.newVariable("Object".toCharArray(), "var9".toCharArray(), buildCharArray(new String[] {
			"new Object() {", 
			"	public String toString() {", 
			"		return \"an object\";", 
			"	}", 
			"}"}));
		GlobalVariable var10 = context.newVariable("Object".toCharArray(), "var10".toCharArray(), null);

		// Install them
		installVariables(10);

		// Get values one by one
		evaluateWithExpectedValue(var1, "1".toCharArray(), "int".toCharArray());
		evaluateWithExpectedValue(var2, "true".toCharArray(), "boolean".toCharArray());
		evaluateWithExpectedValue(var3, "c".toCharArray(), "char".toCharArray());
		evaluateWithExpectedValue(var4, "1.0".toCharArray(), "float".toCharArray());
		evaluateWithExpectedValue(var5, "1.0".toCharArray(), "double".toCharArray());
		evaluateWithExpectedValue(var6, "1".toCharArray(), "short".toCharArray());
		evaluateWithExpectedValue(var7, "1".toCharArray(), "long".toCharArray());
		evaluateWithExpectedValue(var8, "hello".toCharArray(), "java.lang.String".toCharArray());
		evaluateWithExpectedValue(var9, "an object".toCharArray(), "java.lang.Object".toCharArray());
		evaluateWithExpectedValue(var10, "null".toCharArray(), "java.lang.Object".toCharArray());
	} finally {
		// Clean up
		GlobalVariable[] vars = context.allVariables();
		for (int i = 0; i < vars.length; i++) {
			context.deleteVariable(vars[i]);
		}
	}
}
public static Class testClass() {
	return VariableTest.class;
}
/**
 * Tests the assignment of a global variable in a code snippet.
 */
public void testCodeSnippetVarAssign() {
	try {
		// Creates the variables
		GlobalVariable var1 = context.newVariable("int".toCharArray(), "var1".toCharArray(), null);
		GlobalVariable var2 = context.newVariable("boolean".toCharArray(), "var2".toCharArray(), null);
		GlobalVariable var3 = context.newVariable("char".toCharArray(), "var3".toCharArray(), null);
		GlobalVariable var4 = context.newVariable("float".toCharArray(), "var4".toCharArray(), null);
		GlobalVariable var5 = context.newVariable("double".toCharArray(), "var5".toCharArray(), null);
		GlobalVariable var6 = context.newVariable("short".toCharArray(), "var6".toCharArray(), null);
		GlobalVariable var7 = context.newVariable("long".toCharArray(), "var7".toCharArray(), null);
		GlobalVariable var8 = context.newVariable("String".toCharArray(), "var8".toCharArray(), null);
		GlobalVariable var9 = context.newVariable("Object".toCharArray(), "var9".toCharArray(), null);
		GlobalVariable var10 = context.newVariable("Object".toCharArray(), "var10".toCharArray(), null);

		// Install them
		installVariables(10);

		// Assign each of the variable and get its value
		evaluateWithExpectedDisplayString("var1 = 1;".toCharArray(), "1".toCharArray());
		evaluateWithExpectedValue(var1, "1".toCharArray(), "int".toCharArray());
		
		evaluateWithExpectedDisplayString("var2 = true;".toCharArray(), "true".toCharArray());
		evaluateWithExpectedValue(var2, "true".toCharArray(), "boolean".toCharArray());

		evaluateWithExpectedDisplayString("var3 = 'c';".toCharArray(), "c".toCharArray());
		evaluateWithExpectedValue(var3, "c".toCharArray(), "char".toCharArray());
		
		evaluateWithExpectedDisplayString("var4 = (float)1.0;".toCharArray(), "1.0".toCharArray());
		evaluateWithExpectedValue(var4, "1.0".toCharArray(), "float".toCharArray());
		
		evaluateWithExpectedDisplayString("var5 = 1.0;".toCharArray(), "1.0".toCharArray());
		evaluateWithExpectedValue(var5, "1.0".toCharArray(), "double".toCharArray());

		evaluateWithExpectedDisplayString("var6 = (short)1;".toCharArray(), "1".toCharArray());
		evaluateWithExpectedValue(var6, "1".toCharArray(), "short".toCharArray());
		
		evaluateWithExpectedDisplayString("var7 = (long)1;".toCharArray(), "1".toCharArray());
		evaluateWithExpectedValue(var7, "1".toCharArray(), "long".toCharArray());
		
		evaluateWithExpectedDisplayString("var8 = \"hello\";".toCharArray(), "hello".toCharArray());
		evaluateWithExpectedValue(var8, "hello".toCharArray(), "java.lang.String".toCharArray());
		
		evaluateWithExpectedDisplayString(buildCharArray(new String[] {
			"var9 = new Object() {", 
			"	public String toString() {", 
			"		return \"an object\";", 
			"	}", 
			"};"}), "an object".toCharArray());
		evaluateWithExpectedValue(var9, "an object".toCharArray(), "java.lang.Object".toCharArray());

		evaluateWithExpectedDisplayString("var10 = null;".toCharArray(), "null".toCharArray());
		evaluateWithExpectedValue(var10, "null".toCharArray(), "java.lang.Object".toCharArray());
	} finally {
		// Clean up
		GlobalVariable[] vars = context.allVariables();
		for (int i = 0; i < vars.length; i++) {
			context.deleteVariable(vars[i]);
		}
	}
}
/**
 * Tests the retrieval of a global variable from a code snippet.
 */
public void testCodeSnippetVarRetrieval() {
	try {
		// Creates the variables
		GlobalVariable var1 = context.newVariable("int".toCharArray(), "var1".toCharArray(), "1".toCharArray());
		GlobalVariable var2 = context.newVariable("boolean".toCharArray(), "var2".toCharArray(), "true".toCharArray());
		GlobalVariable var3 = context.newVariable("char".toCharArray(), "var3".toCharArray(), "'c'".toCharArray());
		GlobalVariable var4 = context.newVariable("float".toCharArray(), "var4".toCharArray(), "(float)1.0".toCharArray());
		GlobalVariable var5 = context.newVariable("double".toCharArray(), "var5".toCharArray(), "1.0".toCharArray());
		GlobalVariable var6 = context.newVariable("short".toCharArray(), "var6".toCharArray(), "(short)1".toCharArray());
		GlobalVariable var7 = context.newVariable("long".toCharArray(), "var7".toCharArray(), "(long)1".toCharArray());
		GlobalVariable var8 = context.newVariable("String".toCharArray(), "var8".toCharArray(), "\"hello\"".toCharArray());
		GlobalVariable var9 = context.newVariable("Object".toCharArray(), "var9".toCharArray(), buildCharArray(new String[] {
			"new Object() {", 
			"	public String toString() {", 
			"		return \"an object\";", 
			"	}", 
			"}"}));
		GlobalVariable var10 = context.newVariable("Object".toCharArray(), "var10".toCharArray(), null);

		// Install them
		installVariables(10);

		// Get values one by one
		evaluateWithExpectedValue(var1, "1".toCharArray(), "int".toCharArray());
		evaluateWithExpectedValue(var2, "true".toCharArray(), "boolean".toCharArray());
		evaluateWithExpectedValue(var3, "c".toCharArray(), "char".toCharArray());
		evaluateWithExpectedValue(var4, "1.0".toCharArray(), "float".toCharArray());
		evaluateWithExpectedValue(var5, "1.0".toCharArray(), "double".toCharArray());
		evaluateWithExpectedValue(var6, "1".toCharArray(), "short".toCharArray());
		evaluateWithExpectedValue(var7, "1".toCharArray(), "long".toCharArray());
		evaluateWithExpectedValue(var8, "hello".toCharArray(), "java.lang.String".toCharArray());
		evaluateWithExpectedValue(var9, "an object".toCharArray(), "java.lang.Object".toCharArray());
		evaluateWithExpectedValue(var10, "null".toCharArray(), "java.lang.Object".toCharArray());
	} finally {
		// Clean up
		GlobalVariable[] vars = context.allVariables();
		for (int i = 0; i < vars.length; i++) {
			context.deleteVariable(vars[i]);
		}
	}
}
/**
 * Tests variables that include one or more imports.
 */
public void testImports() {
	try {
		// import a package
		context.setImports(new char[][] {"java.io.*".toCharArray()});
		GlobalVariable file = context.newVariable("boolean".toCharArray(), "file".toCharArray(), "new File(\"!@#%\").exists()".toCharArray());
		installVariables(1);
		evaluateWithExpectedValue(file, "false".toCharArray(), "boolean".toCharArray());
		context.deleteVariable(file);

		// import a type
		context.setImports(new char[][] {"java.math.BigInteger".toCharArray()});
		GlobalVariable big = context.newVariable("BigInteger".toCharArray(), "big".toCharArray(), "new BigInteger(\"123456789012345678901234567890\")".toCharArray());
		installVariables(1);
		evaluateWithExpectedValue(big, "123456789012345678901234567890".toCharArray(), "java.math.BigInteger".toCharArray());
		context.deleteVariable(big);

		// import a type and a package
		context.setImports(new char[][] {"java.util.Enumeration".toCharArray(), "java.lang.reflect.*".toCharArray()});
		GlobalVariable fields = context.newVariable("Field[]".toCharArray(), "fields".toCharArray(), "Enumeration.class.getDeclaredFields()".toCharArray());
		installVariables(1);
		evaluateWithExpectedType("return fields;".toCharArray(), "[Ljava.lang.reflect.Field;".toCharArray());
		context.deleteVariable(fields);
	} finally {
		// clean up
		context.setImports(new char[0][]);
	}
}
/**
 * Tests the additions and deletion of variables, installing them each time.
 */
public void testSeveralVariableInstallations() {
	try {
		// Creates 6 variables
		GlobalVariable var1 = context.newVariable("int".toCharArray(), "var1".toCharArray(), "1".toCharArray());
		GlobalVariable var2 = context.newVariable("boolean".toCharArray(), "var2".toCharArray(), "true".toCharArray());
		GlobalVariable var3 = context.newVariable("char".toCharArray(), "var3".toCharArray(), "'c'".toCharArray());
		GlobalVariable var4 = context.newVariable("float".toCharArray(), "var4".toCharArray(), "(float)1.0".toCharArray());
		GlobalVariable var5 = context.newVariable("double".toCharArray(), "var5".toCharArray(), "1.0".toCharArray());
		GlobalVariable var6 = context.newVariable("short".toCharArray(), "var6".toCharArray(), "(short)1".toCharArray());

		// Install the variables
		installVariables(6);

		// Get their values
		evaluateWithExpectedValue(var1, "1".toCharArray(), "int".toCharArray());
		evaluateWithExpectedValue(var2, "true".toCharArray(), "boolean".toCharArray());
		evaluateWithExpectedValue(var3, "c".toCharArray(), "char".toCharArray());
		evaluateWithExpectedValue(var4, "1.0".toCharArray(), "float".toCharArray());
		evaluateWithExpectedValue(var5, "1.0".toCharArray(), "double".toCharArray());
		evaluateWithExpectedValue(var6, "1".toCharArray(), "short".toCharArray());

		// Delete 3 variables
		context.deleteVariable(var2);
		context.deleteVariable(var5);
		context.deleteVariable(var6);
		
		// Install the variables
		installVariables(3);

		// Get their values
		evaluateWithExpectedValue(var1, "1".toCharArray(), "int".toCharArray());
		evaluateWithExpectedValue(var3, "c".toCharArray(), "char".toCharArray());
		evaluateWithExpectedValue(var4, "1.0".toCharArray(), "float".toCharArray());

		// Add 4 more variables
		GlobalVariable var7 = context.newVariable("long".toCharArray(), "var7".toCharArray(), "(long)1".toCharArray());
		GlobalVariable var8 = context.newVariable("String".toCharArray(), "var8".toCharArray(), "\"hello\"".toCharArray());
		GlobalVariable var9 = context.newVariable("Object".toCharArray(), "var9".toCharArray(), buildCharArray(new String[] {
			"new Object() {", 
			"	public String toString() {", 
			"		return \"an object\";", 
			"	}", 
			"}"}));
		GlobalVariable var10 = context.newVariable("Object".toCharArray(), "var10".toCharArray(), null);

		// Install the variables
		installVariables(7);

		// Change value of a variable using a code snippet and move it
		evaluateWithExpectedValue("var3 = 'z'; return var3;".toCharArray(), "z".toCharArray(), "char".toCharArray());
		context.deleteVariable(var3);
		var3 = context.newVariable(var3.getTypeName(), var3.getName(), var3.getInitializer());

		// Change the type of another variable to an incompatible type
		context.deleteVariable(var4);
		var4 = context.newVariable("java.net.URL".toCharArray(), "var4".toCharArray(), "new java.net.URL(\"http://www.ibm.com/index.html\")".toCharArray());

		// Upadte the variables
		installVariables(7);

		// Get their values
		evaluateWithExpectedValue(var1, "1".toCharArray(), "int".toCharArray());
		evaluateWithExpectedValue(var3, "z".toCharArray(), "char".toCharArray());
		evaluateWithExpectedValue(var4, "http://www.ibm.com/index.html".toCharArray(), "java.net.URL".toCharArray());
		evaluateWithExpectedValue(var7, "1".toCharArray(), "long".toCharArray());
		evaluateWithExpectedValue(var8, "hello".toCharArray(), "java.lang.String".toCharArray());
		evaluateWithExpectedValue(var9, "an object".toCharArray(), "java.lang.Object".toCharArray());
		evaluateWithExpectedValue(var10, "null".toCharArray(), "java.lang.Object".toCharArray());
	} finally {
		// Clean up
		GlobalVariable[] vars = context.allVariables();
		for (int i = 0; i < vars.length; i++) {
			context.deleteVariable(vars[i]);
		}
	}
}
}
